/**************************************************************************************

   Copyright (c) Hilscher GmbH. All Rights Reserved.

 **************************************************************************************

   Filename:
    $Workfile: ConnectorConfig.h $
   Last Modification:
    $Author: Robert $
    $Modtime: 28.03.08 14:09 $
    $Revision: 7135 $

   Targets:
     Win32/ANSI   : yes
     Win32/Unicode: no
     WinCE        : no

   Description:
    Definition file of connector configuration functions

   Changes:

     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
     2         27.11.09    SS       Review
     1         28.10.09    SS       created

**************************************************************************************/

/*****************************************************************************/
/*! \file ConnectorConfig.h
*   Definition file of connector configuration functions                     */
/*****************************************************************************/

/* prevent multiple inclusion */
#pragma once

#include <string>
#include <atlstr.h>
#pragma warning(push)
#pragma warning(disable: 4702 4018 4389)
#include <map>
#include <atlrx.h>
#pragma warning(pop)

#include "netXConnectorErrors.h"

/*****************************************************************************/
/*! \addtogroup netX_CONNECTOR_CFG netX Connector configuration
*   \{                                                                       */
/*****************************************************************************/

///////////////////////////////////////////////////////////////////////////////////////////
// Registry keys
///////////////////////////////////////////////////////////////////////////////////////////
#define NXCON_REGKEY_BASE           _T("Software\\Hilscher GmbH\\netXTransport") /*!< Base registry key to store the configuration of the connector */
#define NXCON_MAX_CONFIGURATION_CNT 100                                          /*!< Maximum number of possible configurations per connector */

///////////////////////////////////////////////////////////////////////////////////////////
// Connector config string
///////////////////////////////////////////////////////////////////////////////////////////
#define NXCON_REGEXP_DATASETSPLIT  _T("{[^;]+?};|$")                              /*!< Regular expression to split a configuration string into datasets */
#define NXCON_REGEXP_KEYVALUESPLIT _T("\\b*{[^,;]+?}\\b*=\\b*{[^,;]+?}\\b*,|;|$") /*!< Regular expression to split a dataset into key-value pairs       */
#define NXCON_DATASET_SEPERATOR    _T(";")                                        /*!< Character for seperation of datasets                             */
#define NXCON_KEYVALUE_SEPERATOR   _T(",")                                        /*!< Character for seperation of key-value pairs                      */

///////////////////////////////////////////////////////////////////////////////////////////
// Common connector keys
///////////////////////////////////////////////////////////////////////////////////////////
#define NXCON_COMMON_ENABLED          "ENABLED"           /*!< Configuration key to enable/disable connector            */
#define NXCON_COMMON_KEEPALIVETIMEOUT "KEEPALIVETIMEOUT"  /*!< Configuration key to adjust keep alive timeout           */
#define NXCON_COMMON_RESETTIMEOUT     "RESETTIMEOUT"      /*!< Configuration key to adjust reset timeout                */
#define NXCON_COMMON_SENDTIMEOUT      "SENDTIMEOUT"       /*!< Configuration key to adjust send timeout                 */
#define NXCON_COMMON_EXCLUDE          "EXCLUDE"           /*!< Configuration key to exclude interface                   */
#define NXCON_CONFIGKEY_DEVNAME       "DEVNAME"           /*!< Configuration key holding the interface name             */
#define NXCON_DEVNAME_LAYER           "GLOBAL"            /*!< Configuration value for global layer configuration       */

///////////////////////////////////////////////////////////////////////////////////////////
/// Events for created, changed or removed configuration
///////////////////////////////////////////////////////////////////////////////////////////
typedef enum NX_CONF_KEY_SCOPE_Etag
{
  eINTERFACE,  /*!< Key for Interface configuration */
  eLAYER       /*!< Key for layer configuration     */

} NX_CONF_KEY_SCOPE_E;

///////////////////////////////////////////////////////////////////////////////////////////
/// Map for Key-Value assignment
///////////////////////////////////////////////////////////////////////////////////////////
typedef std::map<CString, CString > CONFIGKEY_MAP;

///////////////////////////////////////////////////////////////////////////////////////////
/// Connector configuration class
///////////////////////////////////////////////////////////////////////////////////////////
class CConnectorConfig
{
public:
  // Constructor
  CConnectorConfig                      ( void);

  CConnectorConfig                      ( const CConnectorConfig& cConnectorConfig);

  CConnectorConfig                      ( const UUID* ptUUID,
                                          const char* szIntfNameToken);

  // Destructor
	~CConnectorConfig                     ( void);

  // Assignment operator
  CConnectorConfig& operator=           ( const CConnectorConfig& cConnectorConfig);

  long            ParseString           ( const char* szConfig);                               /* Parse configuration string     */
  long            CreateString          ( CString& csConfigString);                            /* Create configuration string    */

  CString         CreateInterfaceName   ( void);
  long            CreateInterfaceConfig ( const char* szName);                                 /* Create dataset                 */
  long            RemoveInterfaceConfig ( const char* szName);                                 /* Remove existing dataset        */
  long            LoadInterfaceConfig   ( const char* szName, CONFIGKEY_MAP& cmKeyMap);        /* Obtain key values from dataset */
  long            StoreInterfaceConfig  ( const char* szName, CONFIGKEY_MAP& cmKeyMap);        /* Store datasets key values      */

  long            LoadLayerConfig       ( CONFIGKEY_MAP& cmKeyMap);                            /* Obtain key values from dataset */
  long            StoreLayerConfig      ( CONFIGKEY_MAP& cmKeyMap);                            /* Store datasets key values      */

  long            LoadFromRegistry      ( CString& csRegistryKey);
  long            StoreToRegistry       ( CString& csRegistryKey);

  long            GetDefaultConfig      ( NX_CONF_KEY_SCOPE_E eKeyScope, CONFIGKEY_MAP& cmKeyMap);
  long            GetMaxVal             ( NX_CONF_KEY_SCOPE_E eKeyScope, const char* szKey);
  long            GetMinVal             ( NX_CONF_KEY_SCOPE_E eKeyScope, const char* szKey);

  bool            IsConnectorEnabled    ( void);                                               /* Returns the state of connector */

  /*! Lock Configuration access */
  void            Lock                  ( void) {::EnterCriticalSection(&m_tcsConfigLock);}

  /*! Unlock Configuration access */
  void            Unlock                ( void) {::LeaveCriticalSection(&m_tcsConfigLock);}

  ///////////////////////////////////////////////////////////////////////////////////////////
  /// Map for storing configured interfaces
  ///////////////////////////////////////////////////////////////////////////////////////////
  typedef std::map<CString, CONFIGKEY_MAP > INTFCONFIG_MAP;

  ///////////////////////////////////////////////////////////////////////////////////////////
  /// Iterate over configured interfaces
  ///////////////////////////////////////////////////////////////////////////////////////////
  class iterator
  {

  protected:
    INTFCONFIG_MAP::iterator m_iterIntf;  /*!< Current interface iterator */

  public:

    /*! Default Constructor */
    iterator(INTFCONFIG_MAP::iterator iterIntf):m_iterIntf(iterIntf) {}

    /*! Copy Constructor */
    iterator(const CConnectorConfig::iterator& iterConfig):m_iterIntf(iterConfig.m_iterIntf) {}

    /////////////////////////////////////////////////////////////////////////////
    /// Returns keep alive timeout
    /// \return Keep alive timeout in ms
    /////////////////////////////////////////////////////////////////////////////
    DWORD GetKeepAliveTimeout ( void)
    {
      DWORD dwRet = 0;

      /* Search for timeout config key */
      CONFIGKEY_MAP::iterator iterKey = m_iterIntf->second.find( NXCON_COMMON_KEEPALIVETIMEOUT);
      if (m_iterIntf->second.end() != iterKey)
      {
        dwRet = atoi(iterKey->second);
      }

      return dwRet;
    }

    /////////////////////////////////////////////////////////////////////////////
    /// Returns send timeout
    /// \return Send timeout in ms
    /////////////////////////////////////////////////////////////////////////////
    DWORD GetSendTimeout ( void)
    {
      DWORD dwRet = 0;

      /* Search for timeout config key */
      CONFIGKEY_MAP::iterator iterKey = m_iterIntf->second.find( NXCON_COMMON_SENDTIMEOUT);
      if (m_iterIntf->second.end() != iterKey)
      {
        dwRet = atoi(iterKey->second);
      }

      return dwRet;
    }

    /////////////////////////////////////////////////////////////////////////////
    /// Returns reset timeout
    /// \return Reset timeout in ms
    /////////////////////////////////////////////////////////////////////////////
    DWORD GetResetTimeout ( void)
    {
      DWORD dwRet = 0;

      /* Search for timeout config key */
      CONFIGKEY_MAP::iterator iterKey = m_iterIntf->second.find( NXCON_COMMON_RESETTIMEOUT);
      if (m_iterIntf->second.end() != iterKey)
      {
        dwRet = atoi(iterKey->second);
      }

      return dwRet;
    }

    /////////////////////////////////////////////////////////////////////////////
    /// Returns whether the interface is excluded
    /// \return TRUE if interface is excluded
    /////////////////////////////////////////////////////////////////////////////
    bool IsExcluded ( void)
    {
      bool fRet = false;

      /* Search for exluced config key */
      CONFIGKEY_MAP::iterator iterKey = m_iterIntf->second.find( NXCON_COMMON_EXCLUDE);
      if (m_iterIntf->second.end() != iterKey)
      {
        fRet = atoi(iterKey->second)?true:false;
      }

      return fRet;
    }

    /////////////////////////////////////////////////////////////////////////////
    /// Interface name
    /////////////////////////////////////////////////////////////////////////////
    CString operator*(void)
    {
      return m_iterIntf->first;
    }

    /////////////////////////////////////////////////////////////////////////////
    /// preincrement
    /////////////////////////////////////////////////////////////////////////////
    iterator& operator++(void)
	  {
      ++m_iterIntf;
      return (*this);
	  }

    /////////////////////////////////////////////////////////////////////////////
    /// test for iterator inequality
    /////////////////////////////////////////////////////////////////////////////
		bool operator!=(const iterator& _Right) const
    {
      return this->m_iterIntf != _Right.m_iterIntf;
		}

    /////////////////////////////////////////////////////////////////////////////
    /// test for iterator equality
    /////////////////////////////////////////////////////////////////////////////
    bool operator==(const iterator& _Right) const
    {
      return this->m_iterIntf == _Right.m_iterIntf;
		}
  };

  /////////////////////////////////////////////////////////////////////////////
  /// Find element in configuration via interface name
  /// \return Iterator
  /////////////////////////////////////////////////////////////////////////////
  iterator find( const char* szName)
  {
    return iterator(m_cmIntfConfigMap.find( szName));
  }

  /////////////////////////////////////////////////////////////////////////////
  /// Returns iterator begin
  /// \return Iterator
  /////////////////////////////////////////////////////////////////////////////
  iterator begin()
  {
    return iterator(m_cmIntfConfigMap.begin());
  }

  /////////////////////////////////////////////////////////////////////////////
  /// Returns iterator end
  /// \return Iterator
  /////////////////////////////////////////////////////////////////////////////
  iterator end()
  {
    return iterator(m_cmIntfConfigMap.end());
  }

  /////////////////////////////////////////////////////////////////////////////
  /// Insert interface in configuration
  /// \return Iterator
  /////////////////////////////////////////////////////////////////////////////
  iterator insert( const char* szName)
  {
    iterator iterNew = m_cmIntfConfigMap.end();

    if ( NXCON_NO_ERROR == CreateInterfaceConfig( szName))
    {
      iterNew = m_cmIntfConfigMap.find( szName);
    }

    return iterNew;
  }

protected:
  ///////////////////////////////////////////////////////////////////////////////////////////
  /// Interface control struct
  ///////////////////////////////////////////////////////////////////////////////////////////
  typedef struct KEYCTRL_Ttag
  {
    CString     csDefault;  /*!< Default value      */
    CString     csRegExp;   /*!< Regular expression */
    DWORD       dwMin;      /*!< Minimum value      */
    DWORD       dwMax;      /*!< Maximum value      */

  } KEYCTRL_T, *PKEYCTRL_T;

  ///////////////////////////////////////////////////////////////////////////////////////////
  /// Map for Key-Value assignment
  ///////////////////////////////////////////////////////////////////////////////////////////
  typedef std::map<CString, KEYCTRL_T > KEYCTRL_MAP;

  long            CreateRegistryKey     ( CString& csRegPath, CRegKey& cRegKey);

  long            RegisterConfigKey     ( NX_CONF_KEY_SCOPE_E eKeyScope,
                                          const char*         szKey,
                                          const char*         szDefault,
                                          const char*         szRegExp = NULL);

  long            RegisterConfigKey     ( NX_CONF_KEY_SCOPE_E eKeyScope,
                                          const char*         szKey,
                                          DWORD               dwDefault,
                                          DWORD               dwMin = 0,
                                          DWORD               dwMax = 0);

  BOOL            VerifyValue           ( const char* szValue,
                                          PKEYCTRL_T  ptKeyCtrl);

  BOOL            CheckInterfaceName    ( const char* szInterfaceName);

  UUID                           m_tUUID;                 /*!< UUID of the connector                                           */
  CString                        m_csConfigNameToken;     /*!< Config name token (e.g. IPRANGE, COMPORT )                      */

  INTFCONFIG_MAP                 m_cmIntfConfigMap;       /*!< Map containing configured interfaces with current configuration */
  CONFIGKEY_MAP                  m_cmLayerConfigMap;      /*!< Map containing current layer configuration                      */

private:
  KEYCTRL_MAP                    m_cmIntfKeyCtrlMap;      /*!< Map containing default key values for interface configuration   */
  KEYCTRL_MAP                    m_cmLayerKeyCtrlMap;     /*!< Map containing default key values for layer configuration       */
  CRITICAL_SECTION               m_tcsConfigLock;         /*!< Critical section to lock access to configuration                */
};

///////////////////////////////////////////////////////////////////////////////////////////
/// String parser class
///////////////////////////////////////////////////////////////////////////////////////////
class CConfigString
{
public:
  // Constructor
  CConfigString( const char* szConfig);

  // Destructor
	~CConfigString();

  /*! Get number of datasets in string */
  long  GetDatasetCnt           ( void) { return m_iDatasetCnt;}
  long  Parse                   ( unsigned int   iDatasetIdx,
                                  CONFIGKEY_MAP& cmKeyMap);      /* Factorize string to datasets of key maps */

protected:

  CString           m_csConfigString; /*!< Variable, holding the configuration string                */
  int               m_iDatasetCnt;    /*!< Number of datasets, described by the configuration string */
};

/*****************************************************************************/
/*! \}                                                                       */
/*****************************************************************************/
